<?php

namespace App\Http\Controllers\admin;

use App\Http\Controllers\Controller;
use App\Models\Glosarium;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use Exception;

class GlosariumController extends Controller
{
    public function index()
    {
        $totalIstilah = Glosarium::count();
        $kamus = Glosarium::orderBy('created_at', 'desc')->paginate(10); // tampilkan 10 per halaman
        return view('admin.page.glosarium.index', compact('kamus', 'totalIstilah'));
    }

    public function cari(Request $request)
    {
        $query = $request->input('query');
        $totalIstilah = Glosarium::count();

        $kamus = Glosarium::where('judul', 'like', "%$query%")
                    ->orWhere('deskripsi', 'like', "%$query%")
                    ->paginate(10)
                    ->appends(['query' => $query]);

        return view('user.index', compact('kamus', 'totalIstilah'));
    }

    public function admin(Request $request)
    {
        $query = $request->input('query');
        $totalIstilah = Glosarium::count();
        
        if ($query) {
            // Search dengan scope untuk relevansi yang lebih baik
            $kamus = Glosarium::search($query)
                        ->orderByRelevance($query)
                        ->paginate(10)
                        ->appends(['query' => $query]);
        } else {
            $kamus = Glosarium::orderBy('created_at', 'desc')->paginate(10);
        }
        
        return view('admin.page.glosarium.index', compact('kamus', 'totalIstilah'));
    }

    public function uploadPdfForm()
    {
        // ✅ Perbaikan di sini: kirim $kamus agar tidak error di layout
        $kamus = Glosarium::paginate(10);
        
        // Karena sekarang extraction-only, tidak perlu menampilkan file yang sudah diupload
        return view('admin.page.glosarium.upload_pdf', [
            'kamus' => $kamus,
            'uploadedPdfs' => collect([]) // Empty collection
        ]);
    }

    public function uploadPdf(Request $request)
    {
        $request->validate([
            'pdf' => 'required|mimes:pdf|max:100000',
            'judul_perda' => 'required|string|max:255',
            'link_perda' => 'required|url|max:500',
        ]);

        // Ambil nama asli file (tanpa spasi) - hanya untuk identifikasi
        $originalNameRaw = preg_replace('/\s+/', '_', $request->file('pdf')->getClientOriginalName());

        // Simpan file sementara untuk ekstraksi
        $tempName = time() . '_temp_' . $originalNameRaw;
        
        try {
            $tempPath = $request->file('pdf')->storeAs('temp', $tempName);
            $pdfPath = Storage::path("temp/$tempName");
            
            // Pastikan file benar-benar tersimpan
            if (!file_exists($pdfPath)) {
                return back()->with('error', 'Gagal menyimpan file PDF sementara.');
            }
            
            Log::info('PDF uploaded successfully', ['path' => $pdfPath, 'size' => filesize($pdfPath)]);
            
        } catch (Exception $e) {
            Log::error('Failed to store PDF', ['error' => $e->getMessage()]);
            return back()->with('error', 'Gagal menyimpan file PDF: ' . $e->getMessage());
        }
        
        // Path untuk JSON
        $jsonName = pathinfo($tempName, PATHINFO_FILENAME) . '.json';
        $jsonPath = storage_path("app/public/kamus/$jsonName");
        $scriptPath = base_path("scripts/ekstrak_kamus.py");

        // Pastikan direktori kamus ada
        $kamusDir = dirname($jsonPath);
        if (!is_dir($kamusDir)) {
            try {
                mkdir($kamusDir, 0755, true);
                Log::info('Created directory', ['path' => $kamusDir]);
            } catch (Exception $e) {
                Log::error('Failed to create directory', ['path' => $kamusDir, 'error' => $e->getMessage()]);
                return back()->with('error', 'Gagal membuat direktori output: ' . $e->getMessage());
            }
        }

        // Pastikan direktori dapat ditulis
        if (!is_writable($kamusDir)) {
            Log::error('Directory not writable', ['path' => $kamusDir]);
            return back()->with('error', 'Direktori output tidak dapat ditulis: ' . $kamusDir);
        }

        // Buat command dengan full path Python jika perlu
        $pythonCmd = 'python';  // atau bisa gunakan 'C:\Python\python.exe' jika ada masalah path
        
        // Normalize paths for Windows
        $normalizedPdfPath = str_replace('/', DIRECTORY_SEPARATOR, $pdfPath);
        $normalizedJsonPath = str_replace('/', DIRECTORY_SEPARATOR, $jsonPath);
        $normalizedScriptPath = str_replace('/', DIRECTORY_SEPARATOR, $scriptPath);
        
        $command = "$pythonCmd " . escapeshellarg($normalizedScriptPath) . " " . escapeshellarg($normalizedPdfPath) . " " . escapeshellarg($normalizedJsonPath);
        
        Log::info('Executing Python script', [
            'command' => $command,
            'pdf_path' => $normalizedPdfPath,
            'pdf_exists' => file_exists($normalizedPdfPath),
            'pdf_size' => file_exists($normalizedPdfPath) ? filesize($normalizedPdfPath) : 'N/A',
            'script_exists' => file_exists($normalizedScriptPath),
            'json_dir' => dirname($normalizedJsonPath),
            'json_dir_exists' => is_dir(dirname($normalizedJsonPath)),
            'json_dir_writable' => is_writable(dirname($normalizedJsonPath))
        ]);

        // Eksekusi dengan output capture yang lebih baik
        $output = [];
        $status = 0;
        
        // Set timeout and working directory
        $descriptorspec = [
            0 => ["pipe", "r"],
            1 => ["pipe", "w"],
            2 => ["pipe", "w"]
        ];
        
        $process = proc_open($command, $descriptorspec, $pipes, getcwd());
        
        if (is_resource($process)) {
            // Close stdin
            fclose($pipes[0]);
            
            // Read stdout and stderr
            $stdout = stream_get_contents($pipes[1]);
            $stderr = stream_get_contents($pipes[2]);
            fclose($pipes[1]);
            fclose($pipes[2]);
            
            // Get exit status
            $status = proc_close($process);
            
            // Combine output
            $output = array_filter(explode("\n", trim($stdout . "\n" . $stderr)));
        } else {
            Log::error('Failed to create process for Python script');
            return back()->with('error', 'Gagal menjalankan script ekstraksi PDF.');
        }

        // Hapus file PDF sementara setelah ekstraksi
        if (Storage::exists("temp/$tempName")) {
            Storage::delete("temp/$tempName");
        }

        // Debug output
        Log::info('Python script execution completed', [
            'status' => $status,
            'output' => $output,
            'json_exists' => file_exists($jsonPath)
        ]);

        if (!file_exists($jsonPath)) {
            $errorMsg = 'Ekstraksi gagal atau file tidak ditemukan.';
            if (!empty($output)) {
                $errorMsg .= '<br><strong>Debug Info:</strong><br>' . implode('<br>', $output);
            }
            if ($status !== 0) {
                $errorMsg .= '<br><strong>Exit Code:</strong> ' . $status;
            }
            
            Log::warning('Ekstraksi gagal. File JSON tidak ditemukan.', compact('command', 'status', 'output'));
            return back()->with('error', $errorMsg);
        }

        $jsonContent = file_get_contents($jsonPath);
        $data = json_decode($jsonContent, true);

        if (json_last_error() !== JSON_ERROR_NONE || !is_array($data)) {
            Log::warning('JSON decode error', ['error' => json_last_error_msg(), 'path' => $jsonPath]);
            return back()->with('error', 'Gagal membaca isi file JSON: ' . json_last_error_msg());
        }

        if (empty($data)) {
            return back()->with('error', 'File PDF tidak mengandung istilah yang dapat diekstrak. Pastikan PDF berisi definisi dengan format "istilah adalah..."');
        }

        // Simpan data ke database tanpa path PDF
        $duplicateTerms = []; // Array untuk menyimpan istilah yang duplikat
        $successCount = 0;
        
        foreach ($data as $key => $item) {
            // Handle both old format (key => description) and new format (key => {istilah, definisi})
            if (is_array($item) && isset($item['istilah']) && isset($item['definisi'])) {
                // New format
                $judulOriginal = trim($item['istilah']);
                $deskripsiOriginal = trim($item['definisi']);
            } else {
                // Old format (fallback)
                $judulOriginal = trim($key);
                $deskripsiOriginal = is_string($item) ? trim($item) : '';
            }

            // Skip empty items
            if (empty($judulOriginal) || empty($deskripsiOriginal)) {
                continue;
            }

            // Normalisasi yang kuat untuk menghindari duplikat
            $judulCleaned = preg_replace('/\s+/', ' ', $judulOriginal);
            $judulNormalized = mb_strtolower($judulCleaned);
            $judulFinal = mb_strtoupper($judulCleaned);
            $deskripsiCleaned = preg_replace('/\s+/', ' ', $deskripsiOriginal);

            // Cek duplikasi dengan normalisasi yang sama seperti di store method
            $existing = Glosarium::whereRaw('REPLACE(LOWER(TRIM(judul)), "  ", " ") = ?', [$judulNormalized])->first();

            if ($existing) {
                // Jika ada duplikasi, simpan untuk ditampilkan ke admin
                $duplicateTerms[] = [
                    'judul' => $judulFinal,
                    'existing_desc' => $existing->deskripsi,
                    'new_desc' => $deskripsiCleaned,
                    'existing_id' => $existing->id
                ];
            } else {
                // Tambahkan baru tanpa sumber_pdf
                Glosarium::create([
                    'judul' => $judulFinal,
                    'deskripsi' => $deskripsiCleaned,
                    'sumber_pdf' => null, // Tidak simpan path PDF
                    'judul_perda' => $request->judul_perda,
                    'link_perda' => $request->link_perda,
                ]);
                $successCount++;
            }
        }

        // Hapus file JSON setelah proses (optional)
        if (file_exists($jsonPath)) {
            unlink($jsonPath);
        }

        // Jika ada duplikasi, redirect ke halaman resolusi
        if (!empty($duplicateTerms)) {
            session(['duplicate_terms' => $duplicateTerms]);
            session(['new_perda_data' => [
                'sumber_pdf' => null, // Tidak ada path PDF
                'judul_perda' => $request->judul_perda,
                'link_perda' => $request->link_perda,
            ]]);
            return redirect()->route('admin.master.glosarium.resolve.duplicates')->with('success', 'PDF berhasil diekstrak. Ditemukan ' . count($duplicateTerms) . ' istilah duplikat yang perlu direview.');
        }

        if ($successCount > 0) {
            return redirect()->route('admin.master.glosarium')->with('success', "PDF '{$originalNameRaw}' berhasil diekstrak. Ditemukan {$successCount} istilah baru.");
        } else {
            return back()->with('warning', 'PDF berhasil diproses tetapi tidak ada istilah baru yang ditambahkan (semua mungkin duplikat).');
        }
    }

    public function simpanIstilah(Request $request)
    {
        $judul = $request->input('judul');
        $deskripsi = $request->input('deskripsi');

        // Cek apakah istilah dengan judul yang sama sudah ada
        $istilah = Glosarium::where('judul', $judul)->first();

        if ($istilah) {
            // Jika ada, update dengan deskripsi baru
            $istilah->deskripsi = $deskripsi;
            $istilah->save();
        } else {
            // Jika tidak ada, buat baru
            Glosarium::create([
                'judul' => mb_strtoupper($judul),  // Automatically uppercase
                'deskripsi' => $deskripsi
            ]);
        }

        return redirect()->route('admin.master.glosarium')->with('success', 'Istilah berhasil disimpan atau diperbarui.');
    }

    /**
     * Check if PDF with same name already exists
     */
    public function checkPdfExists(Request $request)
    {
        $filename = $request->input('filename');
        
        if (!$filename) {
            return response()->json(['exists' => false]);
        }
        
        $exists = Glosarium::where('sumber_pdf', 'LIKE', '%' . $filename)
            ->orWhere('sumber_pdf', 'LIKE', '%' . pathinfo($filename, PATHINFO_FILENAME) . '%')
            ->exists();
            
        return response()->json(['exists' => $exists]);
    }

    /**
     * Show duplicate resolution page
     */
    public function resolveDuplicates()
    {
        $duplicateTerms = session('duplicate_terms', []);
        $newPerdaData = session('new_perda_data', []);
        
        if (empty($duplicateTerms)) {
            return redirect()->route('admin.master.glosarium')->with('error', 'Tidak ada duplikasi yang perlu diresolve.');
        }
        
        return view('admin.page.glosarium.resolve_duplicates', compact('duplicateTerms', 'newPerdaData'));
    }

    /**
     * Process duplicate resolution
     */
    public function processDuplicates(Request $request)
    {
        $duplicateTerms = session('duplicate_terms', []);
        $newPerdaData = session('new_perda_data', []);
        $decisions = $request->input('decisions', []);
        
        // Tracking untuk notifikasi
        $stats = [
            'total_processed' => count($duplicateTerms),
            'data_dipertahankan' => 0,
            'data_diganti' => 0,
            'data_duplikat_romawi' => 0,
            'detail_baru' => [],
            'detail_dipertahankan' => []
        ];
        
        foreach ($duplicateTerms as $index => $term) {
            $decision = $decisions[$index] ?? 'keep_existing';
            
            if ($decision === 'replace_existing') {
                // Ganti definisi yang ada dengan yang baru
                $existingRecord = Glosarium::find($term['existing_id']);
                if ($existingRecord) {
                    $existingRecord->update([
                        'deskripsi' => $term['new_desc'],
                        'judul_perda' => $newPerdaData['judul_perda'],
                        'link_perda' => $newPerdaData['link_perda'],
                    ]);
                    $stats['data_diganti']++;
                    $stats['detail_baru'][] = $term['judul'] . ' (diperbarui)';
                }
            } elseif ($decision === 'use_both') {
                // Gunakan semua - simpan yang baru dengan angka romawi
                $romanNumeral = $this->getNextRomanNumeral($term['judul']);
                $newEntry = Glosarium::create([
                    'judul' => mb_strtoupper($term['judul'] . ' ' . $romanNumeral),  // Automatically uppercase
                    'deskripsi' => $term['new_desc'],
                    'sumber_pdf' => null, // Tidak ada PDF
                    'judul_perda' => $newPerdaData['judul_perda'],
                    'link_perda' => $newPerdaData['link_perda'],
                ]);
                $stats['data_duplikat_romawi']++;
                $stats['detail_baru'][] = $term['judul'] . ' ' . $romanNumeral;
                $stats['detail_dipertahankan'][] = $term['judul'] . ' (asli)';
                
            } else { // keep_existing
                $stats['data_dipertahankan']++;
                $stats['detail_dipertahankan'][] = $term['judul'];
            }
        }
        
        // Hapus session data
        session()->forget(['duplicate_terms', 'new_perda_data']);
        
        // Buat pesan notifikasi detail
        $totalBaru = $stats['data_diganti'] + $stats['data_duplikat_romawi'];
        $message = "✅ Proses duplikat selesai!\n\n";
        $message .= "📊 RINGKASAN:\n";
        $message .= "• Total diproses: {$stats['total_processed']} istilah\n";
        $message .= "• Data baru/diperbarui: {$totalBaru} istilah\n";
        $message .= "• Data dipertahankan: {$stats['data_dipertahankan']} istilah\n";
        $message .= "• Data diganti: {$stats['data_diganti']} istilah\n";
        $message .= "• Data duplikat (romawi): {$stats['data_duplikat_romawi']} istilah\n\n";
        
        if (!empty($stats['detail_baru'])) {
            $message .= "🆕 DATA BARU/DIPERBARUI:\n";
            foreach ($stats['detail_baru'] as $item) {
                $message .= "• " . $item . "\n";
            }
            $message .= "\n";
        }
        
        if (!empty($stats['detail_dipertahankan'])) {
            $message .= "💾 DATA DIPERTAHANKAN:\n";
            foreach ($stats['detail_dipertahankan'] as $item) {
                $message .= "• " . $item . "\n";
            }
        }
        
        return redirect()->route('admin.master.glosarium')
            ->with('success_detailed', $message)
            ->with('upload_stats', $stats);
    }

    /**
     * Get next available roman numeral for duplicate terms
     */
    private function getNextRomanNumeral($baseTitle)
    {
        // Roman numerals array - extended to prevent running out
        $romanNumerals = ['I', 'II', 'III', 'IV', 'V', 'VI', 'VII', 'VIII', 'IX', 'X', 'XI', 'XII', 'XIII', 'XIV', 'XV'];
        
        // Normalize base title
        $baseTitle = mb_strtolower(trim($baseTitle));
        
        // Check existing entries with roman numerals
        foreach ($romanNumerals as $numeral) {
            $testTitle = $baseTitle . ' ' . $numeral;
            $existingWithNumeral = Glosarium::whereRaw('LOWER(TRIM(judul)) = ?', [$testTitle])->first();
            if (!$existingWithNumeral) {
                return $numeral;
            }
        }
        
        // If all roman numerals I-XV are taken, use timestamp suffix
        return 'XVI_' . time();
    }

    /**
     * AJAX search untuk live search admin
     */
    public function ajaxSearch(Request $request)
    {
        $query = $request->input('query', '');
        $page = $request->input('page', 1);
        $totalIstilah = Glosarium::count();
        
        if (empty($query)) {
            $kamus = Glosarium::orderBy('created_at', 'desc')->paginate(10, ['*'], 'page', $page);
        } else {
            $kamus = Glosarium::search($query)
                        ->orderByRelevance($query)
                        ->paginate(10, ['*'], 'page', $page);
        }

        // Statistics
        $stats = [
            'total' => $kamus->total(),
            'total_istilah' => $totalIstilah,
            'with_perda' => Glosarium::whereNotNull('link_perda')->count(),
            'from_pdf' => Glosarium::whereNotNull('sumber_pdf')->count(),
            'manual' => Glosarium::whereNull('sumber_pdf')->whereNull('link_perda')->count()
        ];

        return response()->json([
            'data' => $kamus->items(),
            'total' => $kamus->total(), // Total hasil pencarian
            'total_istilah' => $totalIstilah, // Total semua istilah dalam database
            'current_page' => $kamus->currentPage(),
            'last_page' => $kamus->lastPage(),
            'per_page' => $kamus->perPage(),
            'from' => $kamus->firstItem(),
            'to' => $kamus->lastItem(),
            'query' => $query,
            'pagination' => [
                'total' => $kamus->total(),
                'total_istilah' => $totalIstilah,
                'current_page' => $kamus->currentPage(),
                'last_page' => $kamus->lastPage(),
                'per_page' => $kamus->perPage(),
                'from' => $kamus->firstItem(),
                'to' => $kamus->lastItem(),
                'has_pages' => $kamus->hasPages(),
            ],
            'stats' => $stats,
            'query' => $query
        ]);
    }
    public function deleteJson(Request $request)
    {
        $jsonFile = $request->input('json_file');
        
        if (!$jsonFile) {
            return response()->json(['success' => false, 'message' => 'Nama file JSON tidak valid']);
        }
        
        $jsonPath = storage_path('app/public/kamus/' . $jsonFile);
        
        if (file_exists($jsonPath)) {
            unlink($jsonPath);
            return response()->json([
                'success' => true, 
                'message' => "File JSON {$jsonFile} berhasil dihapus"
            ]);
        } else {
            return response()->json(['success' => false, 'message' => 'File JSON tidak ditemukan']);
        }
    }

    /**
     * Show create form for manual entry
     */
    public function create()
    {
        return view('admin.page.glosarium.create');
    }

    /**
     * Store manually created glossary item
     */
    public function store(Request $request)
    {
        $request->validate([
            'judul' => 'required|string|max:255',
            'deskripsi' => 'required|string',
            'judul_perda' => 'nullable|string|max:255',
            'link_perda' => 'nullable|url|max:500',
        ]);

        // Normalisasi yang lebih kuat - bersihkan semua whitespace berlebih
        $judulCleaned = preg_replace('/\s+/', ' ', trim($request->judul));
        $judulNormalized = mb_strtolower($judulCleaned);
        $judulFinal = mb_strtoupper($judulCleaned);
        
        // Bersihkan deskripsi juga
        $deskripsiCleaned = preg_replace('/\s+/', ' ', trim($request->deskripsi));
        
        // Cek duplikasi dengan normalisasi yang sama
        $existing = Glosarium::whereRaw('REPLACE(LOWER(TRIM(judul)), "  ", " ") = ?', [$judulNormalized])->first();
        if ($existing) {
            return back()->withInput()->with('error', 'Istilah dengan judul yang sama sudah ada: "' . $existing->judul . '"');
        }

        Glosarium::create([
            'judul' => $judulFinal,
            'deskripsi' => $deskripsiCleaned,
            'judul_perda' => $request->judul_perda,
            'link_perda' => $request->link_perda,
        ]);

        // Clear any cache if using cache
        \Illuminate\Support\Facades\Cache::forget('glosarium_count');
        \Illuminate\Support\Facades\Cache::flush();

        return redirect()->route('admin.master.glosarium')->with('success', 'Istilah berhasil ditambahkan.');
    }

    /**
     * Show edit form
     */
    public function edit($id)
    {
        $item = Glosarium::findOrFail($id);
        return view('admin.page.glosarium.edit', compact('item'));
    }

    /**
     * Update glossary item
     */
    public function update(Request $request, $id)
    {
        $request->validate([
            'judul' => 'required|string|max:255',
            'deskripsi' => 'required|string',
            'judul_perda' => 'nullable|string|max:255',
            'link_perda' => 'nullable|url|max:500',
        ]);

        $item = Glosarium::findOrFail($id);
        
        // Normalisasi yang lebih kuat - bersihkan semua whitespace berlebih
        $judulCleaned = preg_replace('/\s+/', ' ', trim($request->judul));
        $judulNormalized = mb_strtolower($judulCleaned);
        $judulFinal = mb_strtoupper($judulCleaned);
        
        // Bersihkan deskripsi juga
        $deskripsiCleaned = preg_replace('/\s+/', ' ', trim($request->deskripsi));
        
        // Cek duplikasi kecuali untuk item yang sedang diedit - dengan normalisasi yang sama
        $existing = Glosarium::whereRaw('REPLACE(LOWER(TRIM(judul)), "  ", " ") = ?', [$judulNormalized])
            ->where('id', '!=', $id)
            ->first();
            
        if ($existing) {
            return back()->withInput()->with('error', 'Istilah dengan judul yang sama sudah ada: "' . $existing->judul . '"');
        }

        $item->update([
            'judul' => $judulFinal,
            'deskripsi' => $deskripsiCleaned,
            'judul_perda' => $request->judul_perda,
            'link_perda' => $request->link_perda,
        ]);

        // Clear any cache if using cache
        \Illuminate\Support\Facades\Cache::forget('glosarium_count');
        \Illuminate\Support\Facades\Cache::flush();

        return redirect()->route('admin.master.glosarium')->with('success', 'Istilah berhasil diperbarui.');
    }

    /**
     * Delete glossary item
     */
    public function destroy($id)
    {
        $item = Glosarium::findOrFail($id);
        $item->delete();

        // Clear any cache if using cache
        \Illuminate\Support\Facades\Cache::forget('glosarium_count');
        \Illuminate\Support\Facades\Cache::flush();

        return redirect()->route('admin.master.glosarium')->with('success', 'Istilah berhasil dihapus.');
    }

}