@extends('app-admin')
@section('head')
    @include('admin.partial.head')
    <!-- FontAwesome CDN -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.4/css/all.min.css">
@endsection

@section('content')

<!-- Modern Style CSS -->
<style>
    :root {
        --primary-color: #667eea;
        --primary-gradient: linear-gradient(135deg, #1e3a8a 0%, #3b82f6 50%, #9ca3af 100%);
        --secondary-color: #f093fb;
        --secondary-gradient: linear-gradient(135deg, #f093fb 0%, #f5576c 100%);
        --success-color: #4ecdc4;
        --danger-color: #ff6b6b;
        --warning-color: #feca57;
        --card-shadow: 0 10px 25px rgba(0,0,0,0.1);
        --card-hover-shadow: 0 20px 40px rgba(0,0,0,0.15);
        --border-radius: 15px;
        --transition: all 0.3s cubic-bezier(0.25, 0.8, 0.25, 1);
    }

    .modern-container {
    max-width: 1100px;
    margin-left: 0;
    margin-right: auto;
    padding: 0 0.5rem 1rem 0.5rem;
    }

    .page-header {
    background: var(--primary-gradient);
    border-radius: var(--border-radius);
    padding: 0.5rem 2rem 1rem 2rem; /* lebih rapat ke atas */
    margin-bottom: 0.5rem; /* lebih rapat ke bawah */
    color: white;
    position: relative;
    overflow: hidden;
    }

    .page-header::before {
        content: '';
        position: absolute;
        top: 0;
        right: 0;
        width: 100px;
        height: 100px;
        background: rgba(255,255,255,0.1);
        border-radius: 50%;
        transform: translate(30px, -30px);
    }

    .page-header h1 {
        font-size: 3.5rem;
        font-weight: 800;
        text-shadow: 2px 2px 4px rgba(0,0,0,0.3);
        margin: 0;
        text-shadow: 0 2px 10px rgba(0,0,0,0.2);
    }

    .page-header p {
        margin: 0.5rem 0 0 0;
        opacity: 0.9;
        font-size: 1.1rem;
    }

    .action-bar {
        display: flex;
        gap: 1rem;
        flex-wrap: wrap;
        align-items: center;
        justify-content: space-between;
        margin-bottom: 2rem;
        padding: 1.5rem;
        background: white;
        border-radius: var(--border-radius);
        box-shadow: var(--card-shadow);
    }

    .btn-modern {
        background: var(--secondary-gradient);
        color: white;
        border: none;
        padding: 0.75rem 1.5rem;
        border-radius: 50px;
        font-weight: 600;
        text-decoration: none;
        display: inline-flex;
        align-items: center;
        gap: 0.5rem;
        transition: var(--transition);
        box-shadow: 0 4px 15px rgba(0,0,0,0.1);
    }

    .btn-modern:hover {
        transform: translateY(-2px);
        box-shadow: 0 8px 25px rgba(0,0,0,0.2);
        color: white;
    }

    .btn-upload {
        background: linear-gradient(135deg, #4ecdc4 0%, #44a08d 100%);
    }

    .search-container {
        flex: 1;
        max-width: 500px;
        position: relative;
        transition: var(--transition);
    }

    .google-search-wrapper {
        position: relative;
        width: 100%;
        max-width: 400px;
        margin: 0 auto;
        transition: var(--transition);
    }

    .google-search-wrapper.focused {
        max-width: 600px;
        transform: scale(1.02);
    }

    .search-input {
        width: 100%;
        padding: 12px 20px 12px 50px;
        border: 2px solid #e0e7ff;
        border-radius: 50px;
        background: white;
        font-size: 16px;
        transition: var(--transition);
        box-shadow: 0 2px 8px rgba(0,0,0,0.08);
        outline: none;
    }

    .search-input:focus {
        border-color: var(--primary-color);
        box-shadow: 0 4px 20px rgba(102, 126, 234, 0.15);
        background: #fafbff;
    }

    .search-input::placeholder {
        color: #9ca3af;
        font-style: italic;
    }

    .search-icon {
        position: absolute;
        left: 18px;
        top: 50%;
        transform: translateY(-50%);
        color: #9ca3af;
        font-size: 16px;
        transition: var(--transition);
        pointer-events: none;
        z-index: 2;
    }

    .google-search-wrapper.focused .search-icon {
        color: var(--primary-color);
        transform: translateY(-50%) scale(1.1);
    }

    .search-suggestions {
        position: absolute;
        top: 100%;
        left: 0;
        right: 0;
        background: white;
        border-radius: 12px;
        box-shadow: 0 10px 40px rgba(0,0,0,0.15);
        border: 1px solid #e2e8f0;
        z-index: 1000;
        opacity: 0;
        visibility: hidden;
        transform: translateY(-10px);
        transition: var(--transition);
        margin-top: 8px;
        max-height: 300px;
        overflow-y: auto;
    }

    .search-suggestions.show {
        opacity: 1;
        visibility: visible;
        transform: translateY(0);
    }

    .suggestion-item {
        padding: 12px 20px;
        border-bottom: 1px solid #f1f5f9;
        cursor: pointer;
        transition: var(--transition);
        display: flex;
        align-items: center;
        gap: 12px;
    }

    .suggestion-item:hover {
        background: #f8fafc;
    }

    .suggestion-item:last-child {
        border-bottom: none;
    }

    .suggestion-icon {
        color: #9ca3af;
        font-size: 14px;
        width: 16px;
    }

    .clear-search {
        position: absolute;
        right: 18px;
        top: 50%;
        transform: translateY(-50%);
        background: none;
        border: none;
        color: #9ca3af;
        cursor: pointer;
        font-size: 18px;
        padding: 4px;
        border-radius: 50%;
        transition: var(--transition);
        opacity: 0;
        visibility: hidden;
    }

    .clear-search.show {
        opacity: 1;
        visibility: visible;
    }

    .clear-search:hover {
        background: #f1f5f9;
        color: #64748b;
    }

    .cards-grid {
        display: grid;
        gap: 2rem;
        grid-template-columns: 1fr;
        margin-top: 1.5rem;
    }

    .modern-card {
        background: white;
        border-radius: var(--border-radius);
        padding: 2rem;
        box-shadow: var(--card-shadow);
        transition: var(--transition);
        position: relative;
        overflow: hidden;
        border: 1px solid #f1f5f9;
        display: flex;
        gap: 2rem;
        align-items: flex-start;
        min-height: 160px;
    }

    .modern-card::before {
        content: '';
        position: absolute;
        top: 0;
        left: 0;
        width: 4px;
        height: 100%;
        background: var(--primary-gradient);
        opacity: 0;
        transition: var(--transition);
    }

    .modern-card:hover {
        transform: translateY(-5px);
        box-shadow: var(--card-hover-shadow);
    }

    .modern-card:hover::before {
        opacity: 1;
    }

    .card-main-content {
        flex: 1;
        display: flex;
        flex-direction: column;
        gap: 1rem;
        min-width: 0; /* Allows flex item to shrink */
    }

    .card-actions {
        position: absolute;
        top: 18px;
        right: 18px;
        z-index: 10;
        flex-shrink: 0;
        display: flex;
        align-items: flex-start;
        background: none;
        box-shadow: none;
    }

    .card-header {
        display: flex;
        justify-content: between;
        align-items: flex-start;
        margin-bottom: 0;
    }

    .card-title {
        font-size: 1.35rem;
        font-weight: 700;
        color: #1e293b;
        margin: 0;
        line-height: 1.3;
        word-wrap: break-word;
        overflow-wrap: break-word;
        hyphens: auto;
    }

    .card-content {
        color: #64748b;
        line-height: 1.7;
        margin: 0;
        font-size: 1.05rem;
        display: -webkit-box;
        -webkit-line-clamp: 4;
        line-clamp: 4;
        -webkit-box-orient: vertical;
        overflow: hidden;
        text-overflow: ellipsis;
    }

    .card-footer {
        display: flex;
        justify-content: flex-start;
        align-items: center;
        margin-top: auto;
        padding-top: 0.75rem;
    }

    .source-badge {
        display: inline-flex;
        align-items: center;
        gap: 0.25rem;
        padding: 0.25rem 0.75rem;
        background: #e0f2fe;
        color: #0369a1;
        border-radius: 50px;
        font-size: 0.75rem;
        font-weight: 600;
        text-decoration: none;
        transition: var(--transition);
    }

    .source-badge:hover {
        background: #0369a1;
        color: white;
        text-decoration: none;
    }

    .dropdown-modern {
        position: relative;
    }

    .dropdown-trigger {
        background: #f8fafc;
        border: 1px solid #e2e8f0;
        color: #64748b;
        width: 36px;
        height: 36px;
        border-radius: 50%;
        display: flex;
        align-items: center;
        justify-content: center;
        cursor: pointer;
        transition: var(--transition);
        position: relative;
        z-index: 100;
        pointer-events: auto !important;
    }

    .dropdown-trigger:hover {
        background: #e2e8f0;
        color: #334155;
        transform: scale(1.1);
    }

    .dropdown-menu-modern {
        position: absolute;
        top: 100%;
        right: 0;
        background: white;
        border-radius: 12px;
        box-shadow: 0 10px 40px rgba(0,0,0,0.15);
        border: 1px solid #e2e8f0;
        min-width: 150px;
        overflow: hidden;
        z-index: 9999;
        opacity: 0;
        visibility: hidden;
        transform: translateY(-10px);
        transition: var(--transition);
    }

    /* Fallback untuk dropdown yang terpotong di bawah */
    .dropdown-menu-modern.show-above {
        top: auto;
        bottom: 100%;
        transform: translateY(10px);
    }

    .dropdown-menu-modern.show-above.show {
        transform: translateY(0);
    }

    .dropdown-menu-modern.show {
        opacity: 1;
        visibility: visible;
        transform: translateY(0);
    }

    .dropdown-item-modern {
        display: flex;
        align-items: center;
        gap: 0.75rem;
        padding: 0.75rem 1rem;
        color: #374151;
        text-decoration: none;
        transition: var(--transition);
        border: none;
        background: none;
        width: 100%;
        text-align: left;
        cursor: pointer;
    }

    .dropdown-item-modern:hover {
        background: #f8fafc;
        color: #111827;
    }

    .dropdown-item-modern.edit:hover {
        background: #dbeafe;
        color: #1d4ed8;
    }

    .dropdown-item-modern.delete:hover {
        background: #fef2f2;
        color: #dc2626;
    }

    .dropdown-item-modern i {
        width: 16px;
        flex-shrink: 0;
    }

    .stats-bar {
        display: flex;
        gap: 1rem;
        margin-bottom: 2rem;
    }

    .stat-card {
        background: white;
        padding: 1.5rem;
        border-radius: var(--border-radius);
        box-shadow: var(--card-shadow);
        flex: 1;
        text-align: center;
    }

    .stat-number {
        font-size: 2rem;
        font-weight: 700;
        color: var(--primary-color);
        margin: 0;
    }

    .stat-label {
        color: #64748b;
        font-size: 0.875rem;
        margin: 0.25rem 0 0 0;
    }

    .loading-modern {
        display: flex;
        justify-content: center;
        padding: 2rem;
    }

    .spinner-modern {
        width: 40px;
        height: 40px;
        border: 3px solid #f1f5f9;
        border-top: 3px solid var(--primary-color);
        border-radius: 50%;
        animation: spin 1s linear infinite;
    }

    @keyframes spin {
        0% { transform: rotate(0deg); }
        100% { transform: rotate(360deg); }
    }

    @keyframes fadeInUp {
        0% {
            opacity: 0;
            transform: translateY(30px);
        }
        100% {
            opacity: 1;
            transform: translateY(0);
        }
    }

    .no-results-modern {
        text-align: center;
        padding: 3rem 2rem;
        color: #64748b;
    }

    .no-results-modern i {
        font-size: 4rem;
        color: #cbd5e1;
        margin-bottom: 1rem;
    }

    @media (max-width: 768px) {
        .action-bar {
            flex-direction: column;
            align-items: stretch;
        }

        .search-container {
            max-width: none;
        }

        .modern-card {
            flex-direction: column;
            gap: 1rem;
        }

        .card-actions {
            align-self: flex-end;
        }

        .stats-bar {
            flex-direction: column;
        }

        .page-header h1 {
            font-size: 2.8rem;
        }

        .card-title {
            font-size: 1.1rem;
        }
    }

    @media (max-width: 480px) {
        .modern-container {
            padding: 1rem 0.5rem;
        }

        .page-header {
            padding: 1.5rem;
        }

        .action-bar {
            padding: 1rem;
        }

        .modern-card {
            padding: 1rem;
        }
    }

    /* Modern Pagination Styles for Admin */
    .pagination-wrapper {
        display: flex;
        justify-content: center;
        align-items: center;
        gap: 0.5rem;
        flex-wrap: wrap;
        margin: 2rem 0;
    }

    .pagination-btn {
        background: white;
        border: 2px solid #e2e8f0;
        border-radius: 12px;
        padding: 12px 16px;
        font-weight: 600;
        transition: var(--transition);
        cursor: pointer;
        color: #667eea;
        display: flex;
        align-items: center;
        gap: 0.5rem;
        min-width: 48px;
        justify-content: center;
        box-shadow: 0 2px 8px rgba(0,0,0,0.08);
        font-size: 0.9rem;
    }

    .pagination-btn:hover:not(.disabled):not(.active) {
        background: var(--primary-gradient);
        color: white;
        border-color: #667eea;
        transform: translateY(-2px);
        box-shadow: 0 8px 20px rgba(102, 126, 234, 0.3);
    }

    .pagination-btn.active {
        background: var(--primary-gradient);
        color: white;
        border-color: #667eea;
        transform: scale(1.1);
        box-shadow: 0 8px 20px rgba(102, 126, 234, 0.4);
    }

    .pagination-btn.disabled {
        background: #f8fafc;
        color: #94a3b8;
        border-color: #e2e8f0;
        cursor: not-allowed;
        transform: none;
    }

    .pagination-btn.disabled:hover {
        background: #f8fafc;
        color: #94a3b8;
        border-color: #e2e8f0;
        transform: none;
        box-shadow: 0 2px 8px rgba(0,0,0,0.08);
    }

    .pagination-dots {
        color: #94a3b8;
        font-weight: 600;
        padding: 0 0.5rem;
    }

    .pagination-info {
        color: #64748b;
        font-size: 0.9rem;
        font-weight: 500;
        text-align: center;
        margin-top: 1rem;
        letter-spacing: 0.3px;
    }

    /* Pagination responsiveness */
    @media (max-width: 576px) {
        .pagination-btn {
            padding: 10px 12px;
            font-size: 0.8rem;
            min-width: 44px;
        }
        
        .pagination-wrapper {
            gap: 0.25rem;
        }
        
        .pagination-info {
            font-size: 0.8rem;
        }
    }

    /* Admin Search Feedback Animation */
    @keyframes adminFeedbackSlide {
        0% {
            opacity: 0;
            transform: translateX(-50%) translateY(10px);
        }
        20% {
            opacity: 1;
            transform: translateX(-50%) translateY(0px);
        }
        80% {
            opacity: 1;
            transform: translateX(-50%) translateY(0px);
        }
        100% {
            opacity: 0;
            transform: translateX(-50%) translateY(-10px);
        }
    }

    .admin-search-feedback {
        pointer-events: none;
    }

    /* Admin Search Info Banner Animation */
    @keyframes adminBannerSlideIn {
        0% {
            opacity: 0;
            transform: translateY(-20px);
        }
        100% {
            opacity: 1;
            transform: translateY(0);
        }
    }
</style>

<div class="modern-container" style="margin-top:0!important;padding-top:0!important;">
    <!-- Page Header -->
    <div class="page-header">
        <h1>
            <i class="fas fa-file-alt" style="margin-right: 0.5rem;"></i>
            Admin Dashboard - Glosarium
        </h1>
        <p>Kelola istilah hukum dengan mudah dan efisien</p>
    </div>

    <!-- Action Bar - Selalu tampil -->
    <div class="action-bar">
        <div style="display: flex; gap: 1rem; flex-wrap: wrap;">
            <a href="{{ route('admin.master.glosarium.upload.form') }}" class="btn-modern btn-upload">
                <i class="fas fa-file-pdf"></i>
                Upload PDF
            </a>
            <a href="{{ route('admin.master.glosarium.create') }}" class="btn-modern">
                <i class="fas fa-plus"></i>
                Tambah Manual
            </a>
        </div>

        @if($kamus->count())
        <div class="search-container">
            <div class="google-search-wrapper" id="searchWrapper">
                <i class="fas fa-search search-icon"></i>
                <input type="text" id="modernSearchInput" class="search-input" 
                       placeholder="Cari istilah hukum..." value="{{ request('query') }}">
                <button type="button" class="clear-search" id="clearSearch">
                    <i class="fas fa-times"></i>
                </button>
                <div class="search-suggestions" id="searchSuggestions">
                    <!-- Suggestions will be populated by JavaScript -->
                </div>
            </div>
        </div>
        @endif
    </div>

    @if($kamus->count())
        <!-- Stats Bar -->
        <div class="stats-bar">
            <div class="stat-card">
                <div class="stat-number">{{ isset($totalIstilah) ? $totalIstilah : $kamus->total() }}</div>
                <div class="stat-label">Total Istilah</div>
            </div>
            @if(request('query'))
            <div class="stat-card">
                <div class="stat-number">{{ $kamus->total() }}</div>
                <div class="stat-label">Hasil Ditemukan</div>
            </div>
            @endif
            <div class="stat-card">
                <div class="stat-number">{{ $kamus->currentPage() }}</div>
                <div class="stat-label">Halaman</div>
            </div>
            <div class="stat-card">
                <div class="stat-number">{{ $kamus->lastPage() }}</div>
                <div class="stat-label">Total Halaman</div>
            </div>
        </div>

        <!-- Cards Grid -->
        <div class="cards-grid" id="cardsGrid">
            @foreach($kamus as $item)
            <div class="modern-card" data-title="{{ strtolower($item->judul) }}" 
                 data-description="{{ strtolower($item->deskripsi) }}" style="position:relative;">
                <div class="card-actions">
                    <div class="dropdown-modern">
                        <button class="dropdown-trigger" data-id="{{ $item->id }}">
                            <i class="fas fa-ellipsis-v"></i>
                        </button>
                        <div class="dropdown-menu-modern" id="dropdown-{{ $item->id }}">
                            <a href="{{ route('admin.master.glosarium.edit', $item->id) }}" 
                               class="dropdown-item-modern edit">
                                <i class="fas fa-edit"></i>
                                Edit
                            </a>
                            <button type="button" class="dropdown-item-modern delete delete-btn" 
                                    data-id="{{ $item->id }}" data-title="{{ $item->judul }}">
                                <i class="fas fa-trash"></i>
                                Hapus
                            </button>
                        </div>
                    </div>
                </div>
                <div class="card-main-content">
                    <div class="card-header">
                        <h3 class="card-title">{{ $item->judul }}</h3>
                    </div>

                    <div class="card-content">
                        {{ $item->deskripsi }}
                    </div>

                    <div class="card-footer"> 
                        @if ($item->link_perda)
                            <a href="{{ $item->link_perda }}" target="_blank" class="source-badge">
                                <i class="fas fa-external-link-alt"></i>
                                {{ Str::limit($item->judul_perda ?: 'Lihat Perda', 40) }}
                            </a>
                        @elseif ($item->sumber_pdf)
                            <span class="source-badge">
                                <i class="fas fa-file-pdf"></i>
                                PDF Source
                            </span>
                        @else
                            <span class="source-badge">
                                <i class="fas fa-file-alt"></i>
                                Istilah Hukum
                            </span>
                        @endif
                    </div>
                </div>
            </div>
            @endforeach
        </div>

        <!-- Loading Indicator -->
        <div id="loadingModern" class="loading-modern" style="display: none;">
            <div class="spinner-modern"></div>
        </div>

        <!-- No Results -->
        <div id="noResultsModern" class="no-results-modern" style="display: none;">
            <i class="fas fa-search"></i>
            <h4>Tidak ada hasil ditemukan</h4>
            <p>Coba gunakan kata kunci yang berbeda</p>
        </div>

        <!-- Pagination -->
        <div class="d-flex justify-content-center mt-4" id="paginationContainer">
            {{ $kamus->links() }}
        </div>
    @else
        <div class="no-results-modern">
            <i class="fas fa-book-open"></i>
            <h4>Belum ada data glosarium</h4>
            <p>Gunakan tombol di atas untuk menambah istilah baru atau upload PDF</p>
        </div>
    @endif
</div>

<script>
// Notifikasi Upload Stats
@if(session('success_detailed'))
document.addEventListener('DOMContentLoaded', function() {
    showUploadNotification();
});

function showUploadNotification() {
    const stats = @json(session('upload_stats', []));
    const message = @json(session('success_detailed', ''));
    
    // Create custom notification modal
    const notification = document.createElement('div');
    notification.id = 'upload-notification';
    notification.style.cssText = `
        position: fixed;
        top: 50%;
        left: 50%;
        transform: translate(-50%, -50%);
        background: linear-gradient(135deg, #1e3a8a 0%, #3b82f6 50%, #9ca3af 100%);
        color: white;
        padding: 2rem;
        border-radius: 15px;
        box-shadow: 0 20px 60px rgba(0,0,0,0.3);
        z-index: 10000;
        max-width: 500px;
        width: 90%;
        max-height: 70vh;
        overflow-y: auto;
        animation: slideIn 0.3s ease-out;
    `;
    
    const lines = message.split('\\n');
    let formattedMessage = '';
    
    lines.forEach(line => {
        if (line.includes('📊 RINGKASAN:')) {
            formattedMessage += '<h3 style="margin: 1rem 0; color: #ffd93d;"><i class="fas fa-chart-bar"></i> ' + line.replace('📊 ', '') + '</h3>';
        } else if (line.includes('🆕 DATA BARU')) {
            formattedMessage += '<h4 style="margin: 1rem 0; color: #4ecdc4;"><i class="fas fa-plus-circle"></i> ' + line.replace('🆕 ', '') + '</h4>';
        } else if (line.includes('💾 DATA DIPERTAHANKAN')) {
            formattedMessage += '<h4 style="margin: 1rem 0; color: #a8e6cf;"><i class="fas fa-shield-alt"></i> ' + line.replace('💾 ', '') + '</h4>';
        } else if (line.startsWith('•')) {
            formattedMessage += '<div style="margin: 0.3rem 0; padding-left: 1rem;">• ' + line.substring(1) + '</div>';
        } else if (line.trim()) {
            formattedMessage += '<div style="margin: 0.5rem 0;">' + line + '</div>';
        }
    });
    
    notification.innerHTML = `
        <div style="text-align: center; margin-bottom: 1rem;">
            <i class="fas fa-check-circle" style="font-size: 3rem; color: #4ecdc4; margin-bottom: 1rem;"></i>
            <h2 style="margin: 0; color: white;">Upload Berhasil!</h2>
        </div>
        <div style="text-align: left; line-height: 1.6;">
            ${formattedMessage}
        </div>
        <div style="text-align: center; margin-top: 2rem;">
            <button onclick="closeUploadNotification()" style="
                background: #4ecdc4;
                color: white;
                border: none;
                padding: 0.8rem 2rem;
                border-radius: 25px;
                font-size: 1.1rem;
                font-weight: 600;
                cursor: pointer;
                transition: all 0.3s ease;
            " onmouseover="this.style.background='#45b7aa'" onmouseout="this.style.background='#4ecdc4'">
                <i class="fas fa-times"></i> Tutup
            </button>
        </div>
    `;
    
    // Add animation keyframes
    const style = document.createElement('style');
    style.textContent = `
        @keyframes slideIn {
            from {
                opacity: 0;
                transform: translate(-50%, -60%);
            }
            to {
                opacity: 1;
                transform: translate(-50%, -50%);
            }
        }
        
        #upload-notification::-webkit-scrollbar {
            width: 6px;
        }
        
        #upload-notification::-webkit-scrollbar-track {
            background: rgba(255,255,255,0.1);
            border-radius: 3px;
        }
        
        #upload-notification::-webkit-scrollbar-thumb {
            background: rgba(255,255,255,0.3);
            border-radius: 3px;
        }
    `;
    document.head.appendChild(style);
    
    // Add backdrop
    const backdrop = document.createElement('div');
    backdrop.id = 'notification-backdrop';
    backdrop.style.cssText = `
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background: rgba(0,0,0,0.5);
        z-index: 9999;
        backdrop-filter: blur(3px);
    `;
    
    document.body.appendChild(backdrop);
    document.body.appendChild(notification);
    
    // Auto close after 15 seconds
    setTimeout(() => {
        closeUploadNotification();
    }, 15000);
}

function closeUploadNotification() {
    const notification = document.getElementById('upload-notification');
    const backdrop = document.getElementById('notification-backdrop');
    
    if (notification) {
        notification.style.animation = 'slideOut 0.3s ease-in';
        setTimeout(() => {
            notification.remove();
            if (backdrop) backdrop.remove();
        }, 300);
    }
}

// Add slideOut animation
const slideOutStyle = document.createElement('style');
slideOutStyle.textContent = `
    @keyframes slideOut {
        from {
            opacity: 1;
            transform: translate(-50%, -50%);
        }
        to {
            opacity: 0;
            transform: translate(-50%, -60%);
        }
    }
`;
document.head.appendChild(slideOutStyle);
@endif

// Global variables
let currentDropdown = null;

// Toggle dropdown function
function toggleDropdown(id) {
    console.log('🎯 toggleDropdown called with ID:', id);
    
    const dropdown = document.getElementById(`dropdown-${id}`);
    console.log('🔍 Found dropdown element:', dropdown);
    
    if (!dropdown) {
        console.log('❌ Dropdown element not found for ID:', id);
        return;
    }
    
    // Close any open dropdown
    if (currentDropdown && currentDropdown !== dropdown) {
        console.log('🔄 Closing previous dropdown');
        currentDropdown.classList.remove('show');
        currentDropdown.classList.remove('show-above');
    }
    
    // Remove any previous position classes
    dropdown.classList.remove('show-above');
    
    // Toggle current dropdown
    const isShowing = dropdown.classList.contains('show');
    console.log('📊 Dropdown current state - showing:', isShowing);
    
    dropdown.classList.toggle('show');
    console.log('✅ Dropdown toggled, new classes:', dropdown.className);
    
    // Check if dropdown needs to show above to avoid being cut off
    if (!isShowing && dropdown.classList.contains('show')) {
        const dropdownRect = dropdown.getBoundingClientRect();
        const windowHeight = window.innerHeight;
        
        console.log('📏 Dropdown position check:', {
            bottom: dropdownRect.bottom,
            windowHeight: windowHeight
        });
        
        // If dropdown extends beyond bottom of screen, show it above
        if (dropdownRect.bottom > windowHeight - 20) {
            dropdown.classList.add('show-above');
            console.log('⬆️ Showing dropdown above');
        }
    }
    
    currentDropdown = dropdown.classList.contains('show') ? dropdown : null;
}

// Close dropdown when clicking outside
document.addEventListener('click', function(event) {
    if (!event.target.closest('.dropdown-modern') && currentDropdown) {
        currentDropdown.classList.remove('show');
        currentDropdown.classList.remove('show-above');
        currentDropdown = null;
    }
});

// Modern search functionality - Simplified live search like public version
document.addEventListener('DOMContentLoaded', function() {
    const searchInput = document.getElementById('modernSearchInput');
    const searchWrapper = document.getElementById('searchWrapper');
    const clearSearch = document.getElementById('clearSearch');
    const searchSuggestions = document.getElementById('searchSuggestions');
    const cardsGrid = document.getElementById('cardsGrid');
    const noResults = document.getElementById('noResultsModern');
    const paginationContainer = document.getElementById('paginationContainer');
    
    // Store original cards for filtering
    let originalCards = [];
    let filteredCards = [];
    let currentPage = 1;
    const itemsPerPage = 10;
    
    // Initialize - store all cards data
    if (cardsGrid) {
        const cards = cardsGrid.querySelectorAll('.modern-card');
        originalCards = Array.from(cards).map(card => {
            return {
                element: card.cloneNode(true),
                title: card.getAttribute('data-title') || '',
                description: card.getAttribute('data-description') || '',
                originalElement: card
            };
        });
        filteredCards = [...originalCards];
    }
    
    // Google-like search behavior
    if (searchInput && searchWrapper) {
        let searchTimeout;
        
        // Focus/blur effects
        searchInput.addEventListener('focus', function() {
            searchWrapper.classList.add('focused');
            if (searchInput.value.trim().length > 0) {
                generateSuggestions(searchInput.value.trim());
            }
        });

        searchInput.addEventListener('blur', function(e) {
            setTimeout(() => {
                if (!searchWrapper.contains(e.relatedTarget)) {
                    searchWrapper.classList.remove('focused');
                    hideSuggestions();
                }
            }, 150);
        });

        // Live search on input - PERBAIKAN
        searchInput.addEventListener('input', function() {
            const query = this.value.trim();
            
            clearTimeout(searchTimeout);
            
            if (query.length > 0) {
                clearSearch.classList.add('show');
                generateSuggestions(query);
                
                // Perform live search with debounce
                searchTimeout = setTimeout(() => {
                    performLiveSearch(query);
                }, 300);
            } else {
                clearSearch.classList.remove('show');
                hideSuggestions();
                
                // Jika input dikosongkan, reload halaman untuk kembali ke data asli
                console.log('Input cleared, reloading page');
                setTimeout(() => {
                    window.location.href = "{{ route('admin.master.glosarium') }}";
                }, 100);
            }
        });

        // Tambahan: Event listener untuk keydown (Enter dan Escape)
        searchInput.addEventListener('keydown', function(e) {
            if (e.key === 'Enter') {
                e.preventDefault();
                const query = this.value.trim();
                if (query.length > 0) {
                    // Save query before clearing input
                    const searchQuery = query;
                    
                    // Clear input immediately like public version
                    this.value = '';
                    clearSearch.classList.remove('show');
                    hideSuggestions();
                    
                    // Perform search with saved query
                    performLiveSearch(searchQuery);
                    
                    // Show search feedback (optional)
                    showAdminSearchFeedback(searchQuery);
                } else {
                    window.location.href = "{{ route('admin.master.glosarium') }}";
                }
            } else if (e.key === 'Escape') {
                // ESC key - clear search
                this.value = '';
                clearSearch.classList.remove('show');
                hideSuggestions();
                window.location.href = "{{ route('admin.master.glosarium') }}";
            }
        });

        // Clear search functionality - PERBAIKAN UTAMA
        clearSearch.addEventListener('click', function() {
            console.log('Clear search clicked');
            searchInput.value = '';
            clearSearch.classList.remove('show');
            hideSuggestions();
            
            // Force reload halaman untuk kembali ke data asli
            window.location.href = "{{ route('admin.master.glosarium') }}";
        });

        // Generate search suggestions
        function generateSuggestions(query) {
            if (query.length < 2) {
                hideSuggestions();
                return;
            }

            const suggestions = new Set();
            const queryLower = query.toLowerCase();

            originalCards.forEach(cardData => {
                if (cardData.title.includes(queryLower)) {
                    // Get original title from element
                    const titleElement = cardData.originalElement.querySelector('.card-title');
                    if (titleElement && suggestions.size < 5) {
                        suggestions.add(titleElement.textContent.trim());
                    }
                }
            });

            if (suggestions.size > 0) {
                const suggestionHTML = Array.from(suggestions).map(suggestion => `
                    <div class="suggestion-item" onclick="selectSuggestion('${escapeHtml(suggestion)}')">
                        <i class="fas fa-search suggestion-icon"></i>
                        <span>${escapeHtml(suggestion)}</span>
                    </div>
                `).join('');

                searchSuggestions.innerHTML = suggestionHTML;
                showSuggestions();
            } else {
                hideSuggestions();
            }
        }

        function showSuggestions() {
            if (searchSuggestions.innerHTML.trim()) {
                searchSuggestions.classList.add('show');
            }
        }

        function hideSuggestions() {
            searchSuggestions.classList.remove('show');
        }

        // Global function for suggestion selection
        window.selectSuggestion = function(suggestion) {
            searchInput.value = suggestion;
            clearSearch.classList.add('show');
            hideSuggestions();
            performLiveSearch(suggestion.toLowerCase());
        };

        // Perform live search (AJAX search like user interface) - PERBAIKAN
        function performLiveSearch(query) {
            console.log('performLiveSearch called with query:', query);
            
            // Jika query kosong, langsung redirect ke halaman utama
            if (!query || query.length === 0) {
                console.log('Empty query, redirecting to main page');
                window.location.href = "{{ route('admin.master.glosarium') }}";
                return;
            }
            
            if (isSearching) return;
            
            isSearching = true;
            showLoadingState();

            // Update URL without page reload - always reset to page 1 for new search
            const newUrl = new URL(window.location);
            newUrl.searchParams.delete('page'); // Remove page parameter to reset to page 1
            
            if (query.length > 0) {
                newUrl.searchParams.set('query', query);
            } else {
                newUrl.searchParams.delete('query');
            }
            window.history.pushState({query: query}, '', newUrl);

            // Use AJAX for admin live search
            const apiUrl = "{{ route('admin.master.glosarium.ajax.search') }}";
            const params = new URLSearchParams({
                query: query,
                page: 1  // Always start from page 1 for new search
            });

            fetch(`${apiUrl}?${params}`)
                .then(response => response.json())
                .then(data => {
                    updateSearchResults(data, query);
                    isSearching = false;
                })
                .catch(error => {
                    console.error('Admin search error:', error);
                    isSearching = false;
                    hideLoadingState();
                    // Fallback: reload page with query
                    if (query.length > 0) {
                        window.location.href = `{{ route('admin.master.glosarium') }}?query=${encodeURIComponent(query)}`;
                    } else {
                        window.location.href = `{{ route('admin.master.glosarium') }}`;
                    }
                });
        }
        // Add AJAX search supporting functions
        let isSearching = false;
        
        function showLoadingState() {
            const loadingIndicator = document.getElementById('loadingModern');
            if (loadingIndicator) {
                loadingIndicator.style.display = 'flex';
            }
            if (cardsGrid) {
                cardsGrid.style.opacity = '0.5';
            }
        }
        
        function hideLoadingState() {
            const loadingIndicator = document.getElementById('loadingModern');
            if (loadingIndicator) {
                loadingIndicator.style.display = 'none';
            }
            if (cardsGrid) {
                cardsGrid.style.opacity = '1';
            }
        }

        // Show admin search feedback when Enter is pressed
        function showAdminSearchFeedback(query) {
            const searchContainer = document.querySelector('.search-container');
            if (!searchContainer) return;
            
            // Create feedback element
            const feedback = document.createElement('div');
            feedback.className = 'admin-search-feedback';
            feedback.innerHTML = `
                <div style="
                    position: absolute;
                    top: -45px;
                    left: 50%;
                    transform: translateX(-50%);
                    background: linear-gradient(135deg, #1e3a8a 0%, #3b82f6 50%, #9ca3af 100%);
                    color: white;
                    padding: 8px 16px;
                    border-radius: 20px;
                    font-size: 0.9rem;
                    font-weight: 500;
                    box-shadow: 0 4px 12px rgba(30, 58, 138, 0.3);
                    backdrop-filter: blur(10px);
                    animation: adminFeedbackSlide 3s ease-out forwards;
                    z-index: 1000;
                    white-space: nowrap;
                ">
                    <i class="fas fa-search" style="margin-right: 8px;"></i>
                    Mencari: "${query}"
                </div>
            `;
            
            // Add feedback to container
            searchContainer.style.position = 'relative';
            searchContainer.appendChild(feedback);
            
            // Remove feedback after animation
            setTimeout(() => {
                if (feedback.parentNode) {
                    feedback.parentNode.removeChild(feedback);
                }
            }, 3000);
        }

        // Show search info banner in admin results area
        function showAdminSearchInfoBanner(data, query) {
            // Only show banner if there's a query (from Enter search)
            if (!query || query.trim() === '') {
                // Remove any existing search banner
                const existingBanner = document.querySelector('.admin-search-info-banner');
                if (existingBanner) {
                    existingBanner.remove();
                }
                return;
            }

            const container = document.querySelector('.modern-container');
            if (!container) return;

            // Remove existing banner
            const existingBanner = document.querySelector('.admin-search-info-banner');
            if (existingBanner) {
                existingBanner.remove();
            }

            // Create search info banner
            const banner = document.createElement('div');
            banner.className = 'admin-search-info-banner';
            banner.innerHTML = `
                <div style="
                    background: linear-gradient(135deg, #1e3a8a 0%, #3b82f6 50%, #9ca3af 100%);
                    color: white;
                    padding: 15px 25px;
                    border-radius: 12px;
                    margin: 0 0 25px 0;
                    box-shadow: 0 4px 15px rgba(30, 58, 138, 0.2);
                    border: 1px solid rgba(255,255,255,0.1);
                    display: flex;
                    align-items: center;
                    justify-content: space-between;
                    animation: adminBannerSlideIn 0.4s ease-out;
                ">
                    <div style="display: flex; align-items: center;">
                        <i class="fas fa-search" style="margin-right: 12px; font-size: 1.1rem;"></i>
                        <div>
                            <div style="font-weight: 600; font-size: 1rem;">
                                Hasil pencarian untuk: "${query}"
                            </div>
                            <div style="font-size: 0.85rem; opacity: 0.9; margin-top: 2px;">
                                ${data.total || 0} istilah ditemukan dari ${data.total_istilah || 0} total istilah
                            </div>
                        </div>
                    </div>
                    <button onclick="clearAdminSearch()" style="
                        background: rgba(255,255,255,0.2);
                        border: none;
                        color: white;
                        padding: 8px 12px;
                        border-radius: 6px;
                        cursor: pointer;
                        font-size: 0.85rem;
                        transition: all 0.2s ease;
                    " onmouseover="this.style.background='rgba(255,255,255,0.3)'" 
                       onmouseout="this.style.background='rgba(255,255,255,0.2)'">
                        <i class="fas fa-times" style="margin-right: 5px;"></i>
                        Tampilkan Semua
                    </button>
                </div>
            `;

            // Insert banner after action-bar
            const actionBar = container.querySelector('.action-bar');
            if (actionBar && actionBar.nextElementSibling) {
                container.insertBefore(banner, actionBar.nextElementSibling);
            } else {
                // Fallback: insert at beginning of results
                const cardsGrid = container.querySelector('.cards-grid');
                if (cardsGrid) {
                    container.insertBefore(banner, cardsGrid);
                }
            }
        }

        // Clear admin search function for banner button
        function clearAdminSearch() {
            const searchInput = document.getElementById('liveSearchInput');
            if (searchInput) {
                searchInput.value = '';
            }
            // Reload page to show all results
            window.location.href = "{{ route('admin.master.glosarium') }}";
        }
        
        // Update search results from AJAX response
        function updateSearchResults(data, query) {
            hideLoadingState();
            
            // Show search info banner if there's a query
            showAdminSearchInfoBanner(data, query);
            
            // Update stats
            updateStats(data);
            
            // Update results
            if (data.data && data.data.length > 0) {
                renderResults(data.data, query);
                // Show pagination for live search if more than 1 page
                if (data.last_page > 1) {
                    renderPagination(data, query);
                } else {
                    hidePagination();
                }
            } else {
                renderNoResults(query);
                hidePagination();
            }
        }

        // Update statistics
        function updateStats(data) {
            const statsCards = document.querySelectorAll('.stat-card .stat-number');
            if (statsCards.length >= 4) {
                // Total Istilah (tetap)
                statsCards[0].textContent = data.total_istilah || data.total || 0;
                
                // Hasil ditemukan (hanya jika ada pencarian)
                if (data.query && data.query.length > 0) {
                    if (!document.querySelector('.search-results-stat')) {
                        const searchStat = document.createElement('div');
                        searchStat.className = 'stat-card search-results-stat';
                        searchStat.innerHTML = `
                            <div class="stat-number">${data.total || 0}</div>
                            <div class="stat-label">Hasil Ditemukan</div>
                        `;
                        statsCards[0].closest('.stat-card').after(searchStat);
                    } else {
                        document.querySelector('.search-results-stat .stat-number').textContent = data.total || 0;
                    }
                } else {
                    // Hapus stat hasil pencarian jika tidak ada query
                    const searchStat = document.querySelector('.search-results-stat');
                    if (searchStat) {
                        searchStat.remove();
                    }
                }
                
                // Update halaman stats
                if (statsCards[2]) statsCards[2].textContent = data.current_page || 1;
                if (statsCards[3]) statsCards[3].textContent = data.last_page || 1;
            }
        }

        // Render search results
        function renderResults(items, query) {
            if (!cardsGrid) return;

            let html = '';
            items.forEach((item, index) => {
                const highlightedTitle = highlightText(item.judul, query);
                const highlightedDesc = highlightText(item.deskripsi ? item.deskripsi.substring(0, 150) + '...' : '', query);
                
                // Determine badge and actions
                let badgeContent = '';
                if (item.link_perda) {
                    badgeContent = `
                        <a href="${item.link_perda}" target="_blank" class="source-badge">
                            <i class="fas fa-external-link-alt"></i>
                            ${item.judul_perda ? limitText(item.judul_perda, 40) : 'Lihat Perda'}
                        </a>
                    `;
                } else if (item.sumber_pdf) {
                    badgeContent = `
                        <span class="source-badge">
                            <i class="fas fa-file-pdf"></i>
                            PDF Source
                        </span>
                    `;
                } else {
                    badgeContent = `
                        <span class="source-badge">
                            <i class="fas fa-file-alt"></i>
                            Istilah Hukum
                        </span>
                    `;
                }

                html += `
                    <div class="modern-card" data-aos="fade-up" data-aos-delay="${index * 100}" style="position:relative;">
                        <div class="card-actions">
                            <div class="dropdown-modern">
                                <button type="button" class="dropdown-trigger" data-id="${item.id}">
                                    <i class="fas fa-ellipsis-v"></i>
                                </button>
                                <div class="dropdown-menu-modern" id="dropdown-${item.id}">
                                    <a href="{{ route('admin.master.glosarium.edit', '') }}/${item.id}" class="dropdown-item-modern edit">
                                        <i class="fas fa-edit"></i>
                                        Edit
                                    </a>
                                    <button class="dropdown-item-modern delete delete-btn" data-id="${item.id}" data-title="${escapeHtml(item.judul)}">
                                        <i class="fas fa-trash"></i>
                                        Hapus
                                    </button>
                                </div>
                            </div>
                        </div>
                        <div class="card-main-content">
                            <div class="card-header">
                                <h3 class="card-title">${highlightedTitle}</h3>
                            </div>
                            <div class="card-content">
                                ${highlightedDesc}
                            </div>
                            <div class="card-footer">
                                ${badgeContent}
                            </div>
                        </div>
                    </div>
                `;
            });

            cardsGrid.innerHTML = html;
            
            // Re-attach dropdown handlers
            attachDropdownHandlers();
            
            // Re-attach delete handlers  
            attachDeleteHandlers();
        }

        function renderPagination(data, query) {
            const paginationContainer = document.getElementById('paginationContainer');
            if (!paginationContainer) return;
            
            const currentPage = data.current_page;
            const totalPages = data.last_page;
            
            let paginationHtml = '<nav aria-label="Admin Glosarium pagination">';
            paginationHtml += '<div class="pagination-wrapper">';
            
            // Previous button
            if (currentPage > 1) {
                paginationHtml += `<button class="pagination-btn ajax-page-btn" data-page="${currentPage - 1}" data-query="${query}">
                    <i class="fas fa-chevron-left"></i> Previous
                </button>`;
            } else {
                paginationHtml += `<button class="pagination-btn disabled">
                    <i class="fas fa-chevron-left"></i> Previous
                </button>`;
            }
            
            // Page numbers
            let startPage = Math.max(1, currentPage - 2);
            let endPage = Math.min(totalPages, currentPage + 2);
            
            // Show first page if not in range
            if (startPage > 1) {
                paginationHtml += `<button class="pagination-btn ajax-page-btn" data-page="1" data-query="${query}">1</button>`;
                if (startPage > 2) {
                    paginationHtml += `<span class="pagination-dots">...</span>`;
                }
            }
            
            // Page numbers in range
            for (let i = startPage; i <= endPage; i++) {
                if (i === currentPage) {
                    paginationHtml += `<button class="pagination-btn active">${i}</button>`;
                } else {
                    paginationHtml += `<button class="pagination-btn ajax-page-btn" data-page="${i}" data-query="${query}">${i}</button>`;
                }
            }
            
            // Show last page if not in range
            if (endPage < totalPages) {
                if (endPage < totalPages - 1) {
                    paginationHtml += `<span class="pagination-dots">...</span>`;
                }
                paginationHtml += `<button class="pagination-btn ajax-page-btn" data-page="${totalPages}" data-query="${query}">${totalPages}</button>`;
            }
            
            // Next button
            if (currentPage < totalPages) {
                paginationHtml += `<button class="pagination-btn ajax-page-btn" data-page="${currentPage + 1}" data-query="${query}">
                    Next <i class="fas fa-chevron-right"></i>
                </button>`;
            } else {
                paginationHtml += `<button class="pagination-btn disabled">
                    Next <i class="fas fa-chevron-right"></i>
                </button>`;
            }
            
            paginationHtml += '</div>';
            
            // Add page info
            const startItem = data.from || ((currentPage - 1) * data.per_page) + 1;
            const endItem = data.to || Math.min(currentPage * data.per_page, data.total);
            paginationHtml += `<div class="pagination-info">
                Menampilkan ${startItem} - ${endItem} dari ${data.total} hasil
            </div>`;
            
            paginationHtml += '</nav>';
            
            paginationContainer.innerHTML = paginationHtml;
            paginationContainer.style.display = 'block';
            
            // Add click handlers for AJAX pagination
            const pageButtons = paginationContainer.querySelectorAll('.ajax-page-btn');
            pageButtons.forEach(btn => {
                btn.addEventListener('click', function(e) {
                    e.preventDefault();
                    const page = parseInt(this.getAttribute('data-page'));
                    const query = this.getAttribute('data-query') || '';
                    performAjaxPagination(page, query);
                });
            });
        }
        
        function performAjaxPagination(page, query) {
            if (isSearching) return;
            
            isSearching = true;
            showLoadingState();

            const apiUrl = "{{ route('admin.master.glosarium.ajax.search') }}";
            const params = new URLSearchParams({
                query: query,
                page: page
            });

            fetch(`${apiUrl}?${params}`)
                .then(response => response.json())
                .then(data => {
                    updateSearchResults(data, query);
                    isSearching = false;
                    
                    // Scroll to top of results smoothly
                    cardsGrid.scrollIntoView({ behavior: 'smooth', block: 'start' });
                })
                .catch(error => {
                    console.error('Admin pagination error:', error);
                    isSearching = false;
                    hideLoadingState();
                });
        }
        
        function hidePagination() {
            const paginationContainer = document.getElementById('paginationContainer');
            if (paginationContainer) {
                paginationContainer.style.display = 'none';
            }
        }
        
        // Helper functions
        function highlightText(text, query) {
            if (!query || !text) return text;
            
            const regex = new RegExp(`(${escapeRegExp(query)})`, 'gi');
            return text.replace(regex, '<mark>$1</mark>');
        }

        function limitText(text, length) {
            if (!text) return '';
            return text.length > length ? text.substring(0, length) + '...' : text;
        }
        
        function escapeHtml(text) {
            const div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        }

        function escapeRegExp(string) {
            return string.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
        }
        
        // Global event delegation for dropdown buttons (fallback method)
        document.addEventListener('click', function(e) {
            console.log('🖱️ Global click detected on:', e.target);
            
            // Check if clicked element is dropdown trigger or its child
            const dropdownTrigger = e.target.closest('.dropdown-trigger');
            if (dropdownTrigger) {
                e.preventDefault();
                e.stopPropagation();
                
                console.log('🎯 Dropdown trigger clicked via delegation!');
                
                const id = dropdownTrigger.getAttribute('data-id');
                if (id) {
                    console.log('📝 Processing dropdown for ID:', id);
                    
                    // Visual feedback
                    dropdownTrigger.style.backgroundColor = '#22c55e';
                    setTimeout(() => {
                        dropdownTrigger.style.backgroundColor = '';
                    }, 300);
                    
                    toggleDropdown(id);
                } else {
                    console.log('❌ No data-id found on dropdown trigger');
                }
                return;
            }
            
            // Check for delete button
            const deleteBtn = e.target.closest('.delete-btn');
            if (deleteBtn) {
                e.preventDefault();
                e.stopPropagation();
                
                console.log('🗑️ Delete button clicked via delegation!');
                
                const id = deleteBtn.getAttribute('data-id');
                const title = deleteBtn.getAttribute('data-title');
                
                if (id && confirm(`Apakah Anda yakin ingin menghapus istilah "${title}"?`)) {
                    console.log('✅ Proceeding with delete for ID:', id);
                    
                    // Create and submit form
                    const form = document.createElement('form');
                    form.method = 'POST';
                    form.action = `{{ route('admin.master.glosarium.delete', '') }}/${id}`;
                    
                    const methodField = document.createElement('input');
                    methodField.type = 'hidden';
                    methodField.name = '_method';
                    methodField.value = 'DELETE';
                    
                    const tokenField = document.createElement('input');
                    tokenField.type = 'hidden';
                    tokenField.name = '_token';
                    tokenField.value = '{{ csrf_token() }}';
                    
                    form.appendChild(methodField);
                    form.appendChild(tokenField);
                    document.body.appendChild(form);
                    form.submit();
                }
                return;
            }
            
            // Close all dropdowns if clicking outside
            if (!e.target.closest('.dropdown-modern')) {
                const allDropdowns = document.querySelectorAll('.dropdown-menu-modern');
                allDropdowns.forEach(dropdown => {
                    dropdown.classList.remove('show');
                    dropdown.classList.remove('show-above');
                });
            }
        });
        
        // Re-attach dropdown and delete handlers after AJAX update
        function attachDropdownHandlers() {
            console.log('🔧 Attaching dropdown handlers...');
            
            const dropdownBtns = document.querySelectorAll('.dropdown-trigger');
            console.log('🔍 Found dropdown buttons:', dropdownBtns.length);
            
            dropdownBtns.forEach((btn, index) => {
                console.log(`📝 Processing button ${index + 1}:`, {
                    dataId: btn.getAttribute('data-id'),
                    className: btn.className,
                    offsetParent: btn.offsetParent,
                    visible: window.getComputedStyle(btn).display !== 'none'
                });
                
                // Remove existing onclick if any
                btn.removeAttribute('onclick');
                
                // Add visual test - change color on hover
                btn.style.transition = 'all 0.2s ease';
                
                // Clone the button to remove all previous event listeners
                const newBtn = btn.cloneNode(true);
                btn.parentNode.replaceChild(newBtn, btn);
                
                // Add multiple event types for testing
                ['click', 'mousedown', 'touchstart'].forEach(eventType => {
                    newBtn.addEventListener(eventType, function(e) {
                        e.preventDefault();
                        e.stopPropagation();
                        console.log(`🖱️ ${eventType.toUpperCase()} event on button ${index + 1}!`);
                        
                        // Visual feedback
                        this.style.backgroundColor = '#22c55e';
                        setTimeout(() => {
                            this.style.backgroundColor = '';
                        }, 200);
                        
                        if (eventType === 'click') {
                            const id = this.getAttribute('data-id');
                            if (id) {
                                console.log('🎯 Toggling dropdown for ID:', id);
                                toggleDropdown(id);
                            } else {
                                console.log('❌ No data-id found on button');
                            }
                        }
                    });
                });
                
                // Add hover effect
                newBtn.addEventListener('mouseenter', function() {
                    this.style.backgroundColor = '#e2e8f0';
                    console.log('🖱️ Mouse entered button');
                });
                
                newBtn.addEventListener('mouseleave', function() {
                    this.style.backgroundColor = '';
                });
            });
            
            // Also add global click listener to close dropdowns
            document.addEventListener('click', function(e) {
                if (!e.target.closest('.dropdown-trigger') && !e.target.closest('.dropdown-menu')) {
                    const allDropdowns = document.querySelectorAll('.dropdown-menu');
                    allDropdowns.forEach(dropdown => {
                        dropdown.style.display = 'none';
                    });
                }
            });
            
            console.log('✅ Dropdown handlers attached');
        }
        
        function attachDeleteHandlers() {
            console.log('Attaching delete handlers...');
            
            const deleteButtons = document.querySelectorAll('.delete-btn');
            console.log('Found delete buttons:', deleteButtons.length);
            
            deleteButtons.forEach((button, index) => {
                // Clone the button to remove all previous event listeners
                const newButton = button.cloneNode(true);
                button.parentNode.replaceChild(newButton, button);
                
                newButton.addEventListener('click', function(e) {
                    e.preventDefault();
                    e.stopPropagation();
                    console.log(`Delete button ${index + 1} clicked`);
                    
                    const id = this.getAttribute('data-id');
                    const title = this.getAttribute('data-title');
                    
                    if (!id) {
                        console.log('No data-id found on delete button');
                        return;
                    }
                    
                    if (confirm(`Apakah Anda yakin ingin menghapus istilah "${title}"?`)) {
                        console.log('Deleting item with ID:', id);
                        // Create form and submit
                        const form = document.createElement('form');
                        form.method = 'POST';
                        form.action = `{{ route('admin.master.glosarium.delete', '') }}/${id}`;
                        
                        const methodField = document.createElement('input');
                        methodField.type = 'hidden';
                        methodField.name = '_method';
                        methodField.value = 'DELETE';
                        
                        const tokenField = document.createElement('input');
                        tokenField.type = 'hidden';
                        tokenField.name = '_token';
                        tokenField.value = '{{ csrf_token() }}';
                        
                        form.appendChild(methodField);
                        form.appendChild(tokenField);
                        document.body.appendChild(form);
                        form.submit();
                    }
                });
            });
        }
        
        // Initial attachment of handlers with debugging
        document.addEventListener('DOMContentLoaded', function() {
            console.log('🚀 DOM loaded, setting up handlers...');
            
            // Debug: Count all buttons
            const allButtons = document.querySelectorAll('button');
            console.log('📊 Total buttons found:', allButtons.length);
            
            const dropdownButtons = document.querySelectorAll('.dropdown-trigger');
            console.log('🎯 Dropdown buttons found:', dropdownButtons.length);
            
            const deleteButtons = document.querySelectorAll('.delete-btn');
            console.log('🗑️ Delete buttons found:', deleteButtons.length);
            
            // List all dropdown buttons with their attributes
            dropdownButtons.forEach((btn, index) => {
                console.log(`🔍 Dropdown button ${index + 1}:`, {
                    id: btn.getAttribute('data-id'),
                    class: btn.className,
                    onclick: btn.getAttribute('onclick'),
                    position: btn.getBoundingClientRect(),
                    zIndex: window.getComputedStyle(btn).zIndex,
                    pointerEvents: window.getComputedStyle(btn).pointerEvents
                });
                
                // Add test function to each button
                btn.setAttribute('data-test-index', index);
            });
            
            // Add global test function
            window.testDropdownClick = function(index) {
                const btn = document.querySelector(`[data-test-index="${index}"]`);
                if (btn) {
                    console.log('🧪 Testing manual click on button', index);
                    btn.click();
                } else {
                    console.log('❌ Button not found for index', index);
                }
            };
            
            // Test if elements are actually clickable
            setTimeout(() => {
                console.log('🧪 Testing first dropdown button click programmatically...');
                if (dropdownButtons.length > 0) {
                    const firstBtn = dropdownButtons[0];
                    const rect = firstBtn.getBoundingClientRect();
                    const elementAtPoint = document.elementFromPoint(rect.left + rect.width/2, rect.top + rect.height/2);
                    console.log('📍 Element at button center:', elementAtPoint);
                    console.log('🔍 Is it the button itself?', elementAtPoint === firstBtn || elementAtPoint.closest('.dropdown-trigger') === firstBtn);
                }
            }, 1000);
            
            // Attach handlers
            attachDropdownHandlers();
            attachDeleteHandlers();
            
            console.log('✅ Initial handlers attached');
        });
        
        // Note: Dropdown closing is now handled in attachDropdownHandlers()
        // No additional global click listener needed here
        
        // Update URL without page reload
        function updateURL(query) {
            const newUrl = new URL(window.location);
            if (query.length > 0) {
                newUrl.searchParams.set('query', query);
            } else {
                newUrl.searchParams.delete('query');
            }
            newUrl.searchParams.delete('page'); // Reset to page 1
            window.history.pushState({query: query}, '', newUrl);
        }
        
        // Render filtered results with pagination
        function renderFilteredResults(query) {
            if (!cardsGrid) return;
            
            if (filteredCards.length === 0) {
                renderNoResults(query);
                hidePagination();
                return;
            }
            
            // Calculate pagination
            const totalPages = Math.ceil(filteredCards.length / itemsPerPage);
            const startIndex = (currentPage - 1) * itemsPerPage;
            const endIndex = startIndex + itemsPerPage;
            const currentPageCards = filteredCards.slice(startIndex, endIndex);
            
            // Clear and render current page cards
            let html = '';
            currentPageCards.forEach((cardData, index) => {
                const cardClone = cardData.element.cloneNode(true);
                
                // Highlight search terms in cloned card
                if (query.length > 0) {
                    highlightCardContent(cardClone, query);
                }
                
                // Add animation delay
                cardClone.style.opacity = '0';
                cardClone.style.transform = 'translateY(30px)';
                cardClone.style.animation = `fadeInUp 0.6s ease-out ${index * 100}ms forwards`;
                
                html += cardClone.outerHTML;
            });
            
            cardsGrid.innerHTML = html;
            
            // Update stats
            updateStatsDisplay(query);
            
            // Hide no results
            if (noResults) {
                noResults.style.display = 'none';
            }
            
            // Show pagination if needed
            if (totalPages > 1) {
                renderLivePagination(totalPages, query);
            } else {
                hidePagination();
            }
        }
        
        // Highlight search terms in card content
        function highlightCardContent(cardElement, query) {
            const titleElement = cardElement.querySelector('.card-title');
            const contentElement = cardElement.querySelector('.card-content');
            
            if (titleElement) {
                titleElement.innerHTML = highlightText(titleElement.textContent, query);
            }
            if (contentElement) {
                contentElement.innerHTML = highlightText(contentElement.textContent, query);
            }
        }
        
        // Highlight search text
        function highlightText(text, query) {
            if (!query || query.length === 0) return escapeHtml(text);
            
            const terms = query.toLowerCase().split(' ').filter(term => term.length > 0);
            let highlightedText = escapeHtml(text);
            
            terms.forEach(term => {
                const regex = new RegExp(`(${escapeRegExp(term)})`, 'gi');
                highlightedText = highlightedText.replace(regex, 
                    '<mark style="background: linear-gradient(120deg, #1e3a8a 0%, #3b82f6 50%, #9ca3af 100%); color: white; padding: 2px 6px; border-radius: 4px; font-weight: 600;">$1</mark>'
                );
            });
            
            return highlightedText;
        }
        
        // Update stats display
        function updateStatsDisplay(query) {
            const statsCards = document.querySelectorAll('.stat-card');
            if (statsCards.length >= 2) {
                // Update first stat card (total or filtered count)
                const firstStat = statsCards[0].querySelector('.stat-number');
                const firstLabel = statsCards[0].querySelector('.stat-label');
                
                if (query.length > 0) {
                    firstStat.textContent = filteredCards.length;
                    firstLabel.textContent = 'Hasil Ditemukan';
                } else {
                    firstStat.textContent = originalCards.length;
                    firstLabel.textContent = 'Total Istilah';
                }
                
                // Update page stats if available
                if (statsCards.length >= 3) {
                    const pageStat = statsCards[2].querySelector('.stat-number');
                    pageStat.textContent = currentPage;
                }
                if (statsCards.length >= 4) {
                    const totalPageStat = statsCards[3].querySelector('.stat-number');
                    totalPageStat.textContent = Math.ceil(filteredCards.length / itemsPerPage);
                }
            }
        }
        
        // Render pagination for live search
        function renderLivePagination(totalPages, query) {
            if (!paginationContainer) return;
            
            let paginationHtml = '<nav aria-label="Admin Glosarium pagination">';
            paginationHtml += '<div class="pagination-wrapper">';
            
            // Previous button
            if (currentPage > 1) {
                paginationHtml += `<button class="pagination-btn live-page-btn" data-page="${currentPage - 1}">
                    <i class="fas fa-chevron-left"></i> Previous
                </button>`;
            } else {
                paginationHtml += `<button class="pagination-btn disabled">
                    <i class="fas fa-chevron-left"></i> Previous
                </button>`;
            }
            
            // Page numbers
            let startPage = Math.max(1, currentPage - 2);
            let endPage = Math.min(totalPages, currentPage + 2);
            
            // Show first page if not in range
            if (startPage > 1) {
                paginationHtml += `<button class="pagination-btn live-page-btn" data-page="1">1</button>`;
                if (startPage > 2) {
                    paginationHtml += `<span class="pagination-dots">...</span>`;
                }
            }
            
            // Page numbers in range
            for (let i = startPage; i <= endPage; i++) {
                if (i === currentPage) {
                    paginationHtml += `<button class="pagination-btn active">${i}</button>`;
                } else {
                    paginationHtml += `<button class="pagination-btn live-page-btn" data-page="${i}">${i}</button>`;
                }
            }
            
            // Show last page if not in range
            if (endPage < totalPages) {
                if (endPage < totalPages - 1) {
                    paginationHtml += `<span class="pagination-dots">...</span>`;
                }
                paginationHtml += `<button class="pagination-btn live-page-btn" data-page="${totalPages}">${totalPages}</button>`;
            }
            
            // Next button
            if (currentPage < totalPages) {
                paginationHtml += `<button class="pagination-btn live-page-btn" data-page="${currentPage + 1}">
                    Next <i class="fas fa-chevron-right"></i>
                </button>`;
            } else {
                paginationHtml += `<button class="pagination-btn disabled">
                    Next <i class="fas fa-chevron-right"></i>
                </button>`;
            }
            
            paginationHtml += '</div>';
            
            // Add page info
            const startItem = ((currentPage - 1) * itemsPerPage) + 1;
            const endItem = Math.min(currentPage * itemsPerPage, filteredCards.length);
            paginationHtml += `<div class="pagination-info">
                Menampilkan ${startItem} - ${endItem} dari ${filteredCards.length} hasil
            </div>`;
            
            paginationHtml += '</nav>';
            
            paginationContainer.innerHTML = paginationHtml;
            paginationContainer.style.display = 'block';
            
            // Add click handlers for live pagination
            const pageButtons = paginationContainer.querySelectorAll('.live-page-btn');
            pageButtons.forEach(btn => {
                btn.addEventListener('click', function(e) {
                    e.preventDefault();
                    const page = parseInt(this.getAttribute('data-page'));
                    currentPage = page;
                    renderFilteredResults(searchInput.value.trim());
                    
                    // Scroll to top of results smoothly
                    cardsGrid.scrollIntoView({ behavior: 'smooth', block: 'start' });
                });
            });
        }
        
        function hidePagination() {
            if (paginationContainer) {
                paginationContainer.style.display = 'none';
            }
        }
        
        function renderNoResults(query) {
            if (!cardsGrid) return;

            const message = query.length > 0 
                ? `Pencarian untuk "<strong>${escapeHtml(query)}</strong>" tidak ditemukan.`
                : 'Belum ada istilah tersedia.';
            
            const actionButton = query.length > 0 
                ? `<button onclick="clearLiveSearch()" class="btn-modern" style="margin-top: 1rem;">
                       <i class="fas fa-arrow-left"></i> Tampilkan Semua Data
                   </button>`
                : '';

            cardsGrid.innerHTML = `
                <div class="no-results-modern" style="grid-column: 1 / -1; text-align: center; padding: 3rem 2rem; background: white; border-radius: var(--border-radius); box-shadow: var(--card-shadow);">
                    <i class="fas fa-search" style="font-size: 4rem; color: #cbd5e1; margin-bottom: 1rem;"></i>
                    <h4>Tidak ada hasil ditemukan</h4>
                    <p>${message}</p>
                    ${actionButton}
                </div>
            `;
        }
        
        // Global function to clear search - PERBAIKAN
        window.clearLiveSearch = function() {
            console.log('Global clearLiveSearch called');
            searchInput.value = '';
            clearSearch.classList.remove('show');
            hideSuggestions();
            
            // Force reload halaman untuk kembali ke data asli
            window.location.href = "{{ route('admin.master.glosarium') }}";
        }
        
        // Handle browser back/forward
        window.addEventListener('popstate', function(e) {
            const urlParams = new URLSearchParams(window.location.search);
            const query = urlParams.get('query') || '';
            searchInput.value = query;
            performLiveSearch(query);
        });
        
        // Helper functions
        function escapeHtml(text) {
            const div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        }

        function escapeRegExp(string) {
            return string.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
        }

        // Initialize search if URL has query parameter
        const urlParams = new URLSearchParams(window.location.search);
        const initialQuery = urlParams.get('query');
        if (initialQuery) {
            searchInput.value = initialQuery;
            clearSearch.classList.add('show');
            performLiveSearch(initialQuery);
        }
    }

    // Add smooth animations on load
    const cards = document.querySelectorAll('.modern-card');
    cards.forEach((card, index) => {
        card.style.opacity = '0';
        card.style.transform = 'translateY(20px)';
        
        setTimeout(() => {
            card.style.transition = 'all 0.6s cubic-bezier(0.25, 0.8, 0.25, 1)';
            card.style.opacity = '1';
            card.style.transform = 'translateY(0)';
        }, index * 100);
    });
});
</script>

@endsection